 
#ifndef NVS_H
#define NVS_H

/*
 * All the includes that are needed for code using this module to
 * compile correctly should be #included here.
 */
#include "DX_VOS_BaseTypes.h"

#ifdef __cplusplus
extern "C"
{
#endif

  /*
   *  Object %NVS.h    : %
   *  State           :  %state%
   *  Creation date   :  Tue Nov 23 09:29:46 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief This file containes general math operations that are used by the CRYS software.
   *
   *  \version NVS.h#1:incl:2
   *  \author adams
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************************ Defines ******************************/

/* error definitions */
#define NVS_ERROR_BASE                 0xAB000000UL
#define NVS_MEMORY_ID_NOT_EXIST_ERROR       ( NVS_ERROR_BASE + 0x0UL ) 
#define NVS_MEMORY_ACCESS_FAILURE           ( NVS_ERROR_BASE + 0x1UL )
#define NVS_MEMORY_ID_ALREADY_EXIST_ERROR   ( NVS_ERROR_BASE + 0x2UL )
#define NVS_MEMORY_ALOCC_FAILURE            ( NVS_ERROR_BASE + 0x3UL )
/* This define indicates the maximum error value in NVS */
#define NVS_ERROR_MAX                       ( NVS_ERROR_BASE + 0x10UL )

/* ********************** Type definitions ********************* */

typedef DxUint32_t   DxNvsMemoryId_t;

/* 
  struct that defines the mapping between memory id and file
*/
typedef struct _NVS_Mapping_t
{
  /* validuty of mapping flag. 1 - entry valid, 0 - non-valid */
  DxUint32_t      validFlag;
  
  /* the assosiated file id */
  DxUint32_t      fileId;
  
}NVS_Mapping_t;

/************************ Public Functions **********************/

/* ************** TODO  need to be implemented by the user ********************** */
/**
 * @brief NVS_Init creates a mapping for the fileId input parameter. MemoryId is a token that defines the actual file.
 *        Each application that uses NVS associat the file(fileId) that it is working with , with the some token. 
 *        Application sees only the token, and the mapping between the token and the fileId will be done by NVS
 *        For example, the main storage of the SST may have the memoryId of 1 (that is how SST will see it), and the fileId will be 
 *        file descriptor of the actual file. when NVS recieves request from application , it will translate the memory Id into the file id.
 *
 * @param[in] fileId        - the fileId whose mapping to memory id will be returned in the memoryId_ptr.
 * @param[out] memoryId_ptr          - will hold the allocated memory id.
 *
 * @return DxError_t - On success the value DX_SUCCESS is returned, 
 *                     On failure the following values:
 *
 *                     NVS_MEMORY_ID_ALREADY_EXIST_ERROR - the memory id already exists
 *                     NVS_MEMORY_ALOCC_FAILURE - the buffers that holds the mapping is already full.
 */
DxError_t NVS_InitFile(DxUint32_t       fileId,
                       DxNvsMemoryId_t  memoryId);

/* ************** TODO  need to be implemented by the user ********************** */
/**
 * @brief NVS_GetMemoryId returns the memory Id for the given file id. If mapping does not exists for the given file Id, then error is returned
 * @param[in] fileId        - the fileId whose mapped to memory idis required.
 * @param[out] memoryId_ptr          - will hold the found memory id.
 *
 * @return DxError_t - On success the value DX_SUCCESS is returned, 
 *                     On failure the following values:
 *
 *                     NVS_MEMORY_ID_ALREADY_EXIST_ERROR - the memory id already exists
 *                     NVS_MEMORY_ALOCC_FAILURE - the buffers that holds the mapping is already full.
 */                       
DxError_t NVS_GetMemoryId(DxUint32_t        fileId,
                          DxNvsMemoryId_t*  memoryId_ptr);

/* ************** TODO  need to be implemented by the user ********************** */
/**
 * @brief NVS_Read receives an offset, in bytes, from the beginning of the allocated
 * non-volatile memory space, a destination buffer and its size (also in bytes) 
 * and reads into it from the flash memory. This function returns the number of 
 * Bytes actually read or an Error Code if failed.
 *
 *
 * @param[in] memoryId        - the non volatile memory id - to this memory the NVS driver is accessing.
 * @param[in] offset          - offset, in bytes, from the base address in the flash to be read
 * @param[in] buffer_ptr      - Pointer to the destination buffer
 * @param[in] buff_len        - Number of bytes to read from the momory
 * @param[out] bytes_read_ptr - Number of bytes that are read. 
 *
 * @return DxError_t - On success the value DX_SUCCESS is returned, 
 *                     On failure the following values:
 *
 *                     NVS_MEMORY_ID_NOT_EXIST_ERROR - the memory device doesnt exist.
 *                     NVS_MEMORY_ACCESS_FAILURE - failing to access the memory.
 */
DxError_t NVS_Read(DxNvsMemoryId_t memoryId,       /* the non volatile memory id - to this memory the NVS driver is accessing */
                   DxUint32_t offset,           /*The source address in the non volatile memory where to copy the data from*/
                   char* buffer_ptr,            /* Pointer to destination buffer. */
                   DxUint32_t buff_len,         /* Number of Bytes to read from the Flash */
                   DxUint32_t* bytes_read_ptr); /* Number of bytes that are read */


/* ************** TODO  need to be implemented by the user ********************** */
/**
 * @brief NVS_Write receives an offset, in bytes, from the beginning of the allocated 
 * non-volatile memory space, a source buffer and its size (also in bytes) 
 * and writes it to the flash memory. This function returns the number of bytes actually 
 * written or an Error Code if failed.
 *
 *
 * @param[in] memoryId        - the non volatile memory id - to this memory the NVS driver is accessing.
 * @param[in] offset          - offset, in bytes, from the base address in the flash to be read
 * @param[in] buffer_ptr      - Pointer to the destination buffer
 * @param[in] buff_len        - Number of bytes to write to the memory
 * @param[out] bytesWritten  - Number of bytes written 
 *
 * @return DxError_t - On success the value DX_SUCCESS is returned, 
 *                     On failure the following values:
 *
 *                     NVS_MEMORY_ID_NOT_EXIST_ERROR - the memory device doesnt exist.
 *                     NVS_MEMORY_ACCESS_FAILURE - failing to access the memory.
 */

DxError_t NVS_Write(DxNvsMemoryId_t memoryId,      /* the non volatile memory id - to this memory the NVS driver is accessing */
				    DxUint32_t offset,           /*The source address in the non volatile memory where to copy the data from*/
				    char* buffer_ptr,            /* Pointer to destination buffer. */
				    DxUint32_t buff_len,         /* Number of bytes to write to the memory */
				    DxUint32_t* bytesWritten);   /* Number of bytes that were written */

#ifdef __cplusplus
}
#endif

#endif

